// Code.gs
const CONFIG = {
  SHEET_ID: 'YOUR_GOOGLE_SHEET_ID', // Replace with your Google Sheet ID
  SECRET_KEY: 'YOUR_SECRET_KEY' // Set a secure secret key
};

function doPost(e) {
  return handleRequest(e);
}

function doGet(e) {
  return handleRequest(e);
}

function handleRequest(e) {
  try {
    const action = e.parameter.action;
    
    // Simple authentication (replace with your preferred method)
    if (e.postData) {
      const data = JSON.parse(e.postData.contents);
      if (data.secret !== CONFIG.SECRET_KEY && !isValidRequest(e)) {
        return createResponse({ error: 'Unauthorized' }, 401);
      }
      
      switch (action) {
        case 'getBusinesses':
          return getBusinesses();
        case 'getUsers':
          return getUsers();
        case 'getReferrals':
          return getReferrals();
        case 'getEarnings':
          return getEarnings();
        case 'getSettings':
          return getSettings();
        case 'saveBusiness':
          return saveBusiness(data);
        case 'deleteBusiness':
          return deleteBusiness(data);
        case 'saveSettings':
          return saveSettings(data);
        case 'processPayout':
          return processPayout(data);
        default:
          return createResponse({ error: 'Invalid action' }, 400);
      }
    }
    
    return createResponse({ error: 'Invalid request' }, 400);
  } catch (error) {
    console.error('Error:', error);
    return createResponse({ error: 'Server error: ' + error.message }, 500);
  }
}

function getBusinesses() {
  const sheet = getSheet('Businesses');
  const data = getSheetData(sheet);
  
  const businesses = data.map(row => ({
    id: row[0],
    name: row[1],
    category: row[2],
    phone: row[3],
    location: row[4],
    description: row[5],
    image_url: row[6],
    video_url: row[7],
    featured: row[8] === 'TRUE',
    active: row[9] === 'TRUE',
    created: row[10]
  }));
  
  return createResponse({ success: true, data: businesses });
}

function getUsers() {
  const sheet = getSheet('Users');
  const data = getSheetData(sheet);
  
  const users = data.map(row => ({
    id: row[0],
    name: row[1],
    whatsapp: row[2],
    email: row[3],
    business_name: row[4],
    category: row[5],
    location: row[6],
    refcode: row[7],
    referrer: row[8],
    status: row[9],
    timestamp: row[10]
  }));
  
  return createResponse({ success: true, data: users });
}

function getReferrals() {
  const sheet = getSheet('Referrals');
  const data = getSheetData(sheet);
  
  const referrals = data.map(row => ({
    id: row[0],
    referrer_code: row[1],
    referred_code: row[2],
    status: row[3],
    amount: row[4],
    timestamp: row[5]
  }));
  
  return createResponse({ success: true, data: referrals });
}

function getEarnings() {
  const sheet = getSheet('Earnings');
  const data = getSheetData(sheet);
  
  const earnings = data.map(row => ({
    id: row[0],
    user_id: row[1],
    amount: row[2],
    type: row[3],
    status: row[4],
    timestamp: row[5]
  }));
  
  return createResponse({ success: true, data: earnings });
}

function getSettings() {
  const sheet = getSheet('Settings');
  const data = getSheetData(sheet);
  
  const settings = {};
  data.forEach(row => {
    if (row[0] && row[1]) {
      settings[row[0]] = row[1];
    }
  });
  
  return createResponse({ success: true, data: settings });
}

function saveBusiness(data) {
  try {
    const sheet = getSheet('Businesses');
    const id = data.id || generateId();
    
    if (data.id) {
      // Update existing business
      const rows = sheet.getDataRange().getValues();
      const header = rows[0];
      const idIndex = header.indexOf('id');
      
      for (let i = 1; i < rows.length; i++) {
        if (rows[i][idIndex] === data.id) {
          sheet.getRange(i + 1, 2, 1, 10).setValues([[
            data.name,
            data.category,
            data.phone,
            data.location,
            data.description,
            data.image_url,
            data.video_url,
            data.featured,
            data.active,
            new Date().toISOString()
          ]]);
          break;
        }
      }
    } else {
      // Add new business
      sheet.appendRow([
        id,
        data.name,
        data.category,
        data.phone,
        data.location,
        data.description,
        data.image_url,
        data.video_url,
        data.featured,
        data.active,
        new Date().toISOString()
      ]);
    }
    
    return createResponse({ success: true, id: id });
  } catch (error) {
    return createResponse({ success: false, error: error.message });
  }
}

function deleteBusiness(data) {
  try {
    const sheet = getSheet('Businesses');
    const rows = sheet.getDataRange().getValues();
    const header = rows[0];
    const idIndex = header.indexOf('id');
    
    for (let i = 1; i < rows.length; i++) {
      if (rows[i][idIndex] === data.id) {
        sheet.deleteRow(i + 1);
        break;
      }
    }
    
    return createResponse({ success: true });
  } catch (error) {
    return createResponse({ success: false, error: error.message });
  }
}

function saveSettings(data) {
  try {
    const sheet = getSheet('Settings');
    sheet.clear();
    
    // Add headers
    sheet.getRange(1, 1, 1, 2).setValues([['key', 'value']]);
    
    // Add settings
    const settingsData = Object.entries(data).map(([key, value]) => [key, value]);
    if (settingsData.length > 0) {
      sheet.getRange(2, 1, settingsData.length, 2).setValues(settingsData);
    }
    
    return createResponse({ success: true });
  } catch (error) {
    return createResponse({ success: false, error: error.message });
  }
}

function processPayout(data) {
  try {
    const sheet = getSheet('Referrals');
    const rows = sheet.getDataRange().getValues();
    const header = rows[0];
    const idIndex = header.indexOf('id');
    const statusIndex = header.indexOf('status');
    
    for (let i = 1; i < rows.length; i++) {
      if (rows[i][idIndex] === data.id) {
        sheet.getRange(i + 1, statusIndex + 1).setValue('paid');
        break;
      }
    }
    
    return createResponse({ success: true });
  } catch (error) {
    return createResponse({ success: false, error: error.message });
  }
}

// Helper functions
function getSheet(sheetName) {
  const spreadsheet = SpreadsheetApp.openById(CONFIG.SHEET_ID);
  let sheet = spreadsheet.getSheetByName(sheetName);
  
  if (!sheet) {
    sheet = spreadsheet.insertSheet(sheetName);
    
    // Initialize with headers based on sheet type
    switch (sheetName) {
      case 'Businesses':
        sheet.getRange('A1:K1').setValues([[
          'id', 'name', 'category', 'phone', 'location', 'description', 
          'image_url', 'video_url', 'featured', 'active', 'created'
        ]]);
        break;
      case 'Users':
        sheet.getRange('A1:K1').setValues([[
          'id', 'name', 'whatsapp', 'email', 'business_name', 'category',
          'location', 'refcode', 'referrer', 'status', 'timestamp'
        ]]);
        break;
      case 'Referrals':
        sheet.getRange('A1:F1').setValues([[
          'id', 'referrer_code', 'referred_code', 'status', 'amount', 'timestamp'
        ]]);
        break;
      case 'Earnings':
        sheet.getRange('A1:F1').setValues([[
          'id', 'user_id', 'amount', 'type', 'status', 'timestamp'
        ]]);
        break;
      case 'Settings':
        sheet.getRange('A1:B1').setValues([['key', 'value']]);
        break;
    }
  }
  
  return sheet;
}

function getSheetData(sheet) {
  const data = sheet.getDataRange().getValues();
  return data.slice(1); // Remove header row
}

function generateId() {
  return Utilities.getUuid();
}

function createResponse(data, statusCode = 200) {
  return ContentService
    .createTextOutput(JSON.stringify(data))
    .setMimeType(ContentService.MimeType.JSON)
    .setStatusCode(statusCode);
}

function isValidRequest(e) {
  // Add your request validation logic here
  // This could include IP whitelisting, token validation, etc.
  return true;
}